--[[
    RealGPSConfig

    Adds the RealGPS Configuration to all vehicles.

	@author: 		BayernGamers
	@date: 			20.05.2025
	@version:		1.0

	History:		v1.0 @20.05.2025 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------
                    
	
	License:        Terms:
                      
    Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "RealGPSConfig.lua")

RealGPSConfig = {}
RealGPSConfig.MOD_DIR = g_currentModDirectory
RealGPSConfig.MOD_NAME = g_currentModName
RealGPSConfig.BASE_PATH = "vehicle.realGPS"
RealGPSConfig.DEFAULT_VEHICLE_XML = RealGPSConfig.MOD_DIR .. "xml/defaultVehicles.xml"
RealGPSConfig.DEFAULT_VEHICLE_BASE_PATH = "defaultVehicles.vehicle(?)"
RealGPSConfig.PRICE = 500

function RealGPSConfig.checkIsDefaultVehicle(xmlFile)
    local defaultVehiclesXML = XMLFile.loadIfExists("realGPSDefaultVehicles", RealGPSConfig.DEFAULT_VEHICLE_XML, RealGPSConfig.DEFAULT_VEHICLE_SCHEMA)

    local found = false
    local foundIndex = -1
    if defaultVehiclesXML ~= nil then
        defaultVehiclesXML:iterate(RealGPSConfig.DEFAULT_VEHICLE_BASE_PATH:sub(1, -4), function (index, key)
            local filename = defaultVehiclesXML:getString(key .. "#filename")
            
            local pattern = ".*pdlc"
            local dlcFilename = nil
            if filename:find("pdlc") ~= nil then
                dlcFilename = "pdlc" .. filename:gsub(pattern, "")
            end
            
            
            local xmlDLCFilename = nil
            if xmlFile.filename:find("pdlc") ~= nil then
                xmlDLCFilename = "pdlc" .. xmlFile.filename:gsub(pattern, "")
            end
            
            filename = Utils.getFilename(filename)
            if xmlFile.filename == filename or (dlcFilename ~= nil and xmlDLCFilename ~= nil and xmlDLCFilename == dlcFilename) then
                found = true
                foundIndex = index
            end
        end)
    end

    return found, foundIndex
end

function RealGPSConfig.isPreparedForRealGPS(xmlFile, customEnv)
    local isPrepared = false
    local mod = customEnv ~= nil and g_modManager:getModByName(customEnv) or nil
    local baseDirectory = mod ~= nil and mod.modDir or nil

    if xmlFile ~= nil and xmlFile:hasProperty(RealGPSConfig.BASE_PATH) then
        isPrepared = true
    end

    if xmlFile ~= nil and xmlFile:hasProperty("vehicle.dashboard.compounds") then
        xmlFile:iterate("vehicle.dashboard.compounds.compound", function (_, compoundKey)
           local filename = xmlFile:getValue(compoundKey .. "#filename")
           local compoundName = xmlFile:getValue(compoundKey .. "#name")

           if filename ~= nil and filename ~= "" and filename ~= " " then
               local dashboardXMLFilename = Utils.getFilename(filename, baseDirectory)
                local dashboardXMLFile = XMLFile.load("dashboardCompoundsXML", dashboardXMLFilename, Dashboard.compoundsXMLSchema)

                if dashboardXMLFile ~= nil then
                    dashboardXMLFile:iterate("dashboardCompounds.dashboardCompound", function(_, key2)
                        if dashboardXMLFile:getValue(key2 .. "#name") == compoundName then
                            if dashboardXMLFile:hasProperty(key2 .. ".realGPS") then
                                isPrepared = true
                            end
                        end
                    end)

                    dashboardXMLFile:delete()
                end
           end
        end)
    end

    return isPrepared
end

function RealGPSConfig.getConfigurationsFromXML(configurationManager, superFunc, xmlFile, baseKey, configs, customEnv, isMod, storeItem)
    local configurations, defaultConfigIds = superFunc(configurationManager, xmlFile, baseKey, configs, customEnv, isMod, storeItem)
    local isDefaultVehicle, _ = RealGPSConfig.checkIsDefaultVehicle(xmlFile)
    local isPrepared = RealGPSConfig.isPreparedForRealGPS(xmlFile, customEnv)

    -- Overwrite propably user defined configurations as RealGPS is not intended to support custom configs.
    if configurations ~= nil and configurations["realGPS"] ~= nil then
        configurations["realGPS"] = nil
    end

    if isDefaultVehicle or isPrepared then
        configurations = configurations == nil and {} or configurations
        defaultConfigIds = defaultConfigIds == nil and {} or defaultConfigIds
        local realGPSConfigs = {}

        local realGPSNo = VehicleConfigurationItem.new("realGPS")
        realGPSNo.name = g_i18n:getText("configuration_valueNo")
        realGPSNo.index = 1
        realGPSNo.saveId = "1"
        realGPSNo.price = 0
        realGPSNo.isYesNoOption = true
        table.insert(realGPSConfigs, realGPSNo)

        local realGPSYes = VehicleConfigurationItem.new("realGPS")
        realGPSYes.isDefault = true
        realGPSYes.name = g_i18n:getText("configuration_valueYes")
        realGPSYes.index = 2
        realGPSYes.saveId = "2"
        realGPSYes.price = RealGPSConfig.PRICE
        realGPSYes.isYesNoOption = true
        table.insert(realGPSConfigs, realGPSYes)

        defaultConfigIds.realGPS = ConfigurationUtil.getDefaultConfigIdFromItems(realGPSConfigs)
        configurations.realGPS = realGPSConfigs
    end

    return configurations, defaultConfigIds
end

ConfigurationUtil.getConfigurationsFromXML = Utils.overwrittenFunction(ConfigurationUtil.getConfigurationsFromXML, RealGPSConfig.getConfigurationsFromXML)